/*
 * Copyright (c) 2017, 2025, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package java.lang.runtime;

import java.lang.classfile.ClassFile;
import java.lang.classfile.ClassHierarchyResolver;
import java.lang.classfile.Opcode;
import java.lang.constant.ClassDesc;
import java.lang.constant.MethodTypeDesc;
import java.lang.invoke.ConstantCallSite;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodType;
import java.lang.invoke.StringConcatFactory;
import java.lang.invoke.TypeDescriptor;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Objects;

import static java.lang.classfile.ClassFile.ACC_STATIC;
import static java.lang.constant.ConstantDescs.*;
import static java.util.Objects.requireNonNull;

/**
 * Bootstrap methods for state-driven implementations of core methods,
 * including {@link Object#equals(Object)}, {@link Object#hashCode()}, and
 * {@link Object#toString()}.  These methods may be used, for example, by
 * Java compiler implementations to implement the bodies of {@link Object}
 * methods for record classes.
 *
 * @since 16
 */
public final class ObjectMethods {

    private ObjectMethods() { }

    private static final int MAX_STRING_CONCAT_SLOTS = 20;

    private static final MethodHandle FALSE = MethodHandles.zero(boolean.class);
    private static final MethodHandle TRUE = MethodHandles.constant(boolean.class, true);
    private static final MethodHandle ZERO = MethodHandles.zero(int.class);
    private static final MethodHandle CLASS_IS_INSTANCE;
    private static final MethodHandle IS_NULL;
    private static final MethodHandle IS_ARG0_NULL;
    private static final MethodHandle IS_ARG1_NULL;
    private static final MethodHandle OBJECT_EQ;
    private static final MethodHandle HASH_COMBINER;
    private static final MethodType MT_OBJECT_BOOLEAN = MethodType.methodType(boolean.class, Object.class);
    private static final MethodType MT_INT = MethodType.methodType(int.class);
    private static final MethodTypeDesc MTD_OBJECT_BOOLEAN = MethodTypeDesc.of(CD_boolean, CD_Object);
    private static final MethodTypeDesc MTD_INT = MethodTypeDesc.of(CD_int);

    private static final HashMap<Class<?>, MethodHandle> primitiveEquals = new HashMap<>();
    private static final HashMap<Class<?>, MethodHandle> primitiveHashers = new HashMap<>();

    static {
        try {
            Class<ObjectMethods> OBJECT_METHODS_CLASS = ObjectMethods.class;
            MethodHandles.Lookup publicLookup = MethodHandles.publicLookup();
            MethodHandles.Lookup lookup = MethodHandles.lookup();

            CLASS_IS_INSTANCE = publicLookup.findVirtual(Class.class, "isInstance",
                                                         MethodType.methodType(boolean.class, Object.class));

            var objectsIsNull = publicLookup.findStatic(Objects.class, "isNull",
                                                        MethodType.methodType(boolean.class, Object.class));
            IS_NULL = objectsIsNull;
            IS_ARG0_NULL = MethodHandles.dropArguments(objectsIsNull, 1, Object.class);
            IS_ARG1_NULL = MethodHandles.dropArguments(objectsIsNull, 0, Object.class);

            OBJECT_EQ = lookup.findStatic(OBJECT_METHODS_CLASS, "eq",
                                          MethodType.methodType(boolean.class, Object.class, Object.class));
            HASH_COMBINER = lookup.findStatic(OBJECT_METHODS_CLASS, "hashCombiner",
                                              MethodType.methodType(int.class, int.class, int.class));

            primitiveEquals.put(byte.class, lookup.findStatic(OBJECT_METHODS_CLASS, "eq",
                                                              MethodType.methodType(boolean.class, byte.class, byte.class)));
            primitiveEquals.put(short.class, lookup.findStatic(OBJECT_METHODS_CLASS, "eq",
                                                               MethodType.methodType(boolean.class, short.class, short.class)));
            primitiveEquals.put(char.class, lookup.findStatic(OBJECT_METHODS_CLASS, "eq",
                                                              MethodType.methodType(boolean.class, char.class, char.class)));
            primitiveEquals.put(int.class, lookup.findStatic(OBJECT_METHODS_CLASS, "eq",
                                                             MethodType.methodType(boolean.class, int.class, int.class)));
            primitiveEquals.put(long.class, lookup.findStatic(OBJECT_METHODS_CLASS, "eq",
                                                              MethodType.methodType(boolean.class, long.class, long.class)));
            primitiveEquals.put(float.class, lookup.findStatic(OBJECT_METHODS_CLASS, "eq",
                                                               MethodType.methodType(boolean.class, float.class, float.class)));
            primitiveEquals.put(double.class, lookup.findStatic(OBJECT_METHODS_CLASS, "eq",
                                                                MethodType.methodType(boolean.class, double.class, double.class)));
            primitiveEquals.put(boolean.class, lookup.findStatic(OBJECT_METHODS_CLASS, "eq",
                                                                 MethodType.methodType(boolean.class, boolean.class, boolean.class)));

            primitiveHashers.put(byte.class, lookup.findStatic(Byte.class, "hashCode",
                                                               MethodType.methodType(int.class, byte.class)));
            primitiveHashers.put(short.class, lookup.findStatic(Short.class, "hashCode",
                                                                MethodType.methodType(int.class, short.class)));
            primitiveHashers.put(char.class, lookup.findStatic(Character.class, "hashCode",
                                                               MethodType.methodType(int.class, char.class)));
            primitiveHashers.put(int.class, lookup.findStatic(Integer.class, "hashCode",
                                                              MethodType.methodType(int.class, int.class)));
            primitiveHashers.put(long.class, lookup.findStatic(Long.class, "hashCode",
                                                               MethodType.methodType(int.class, long.class)));
            primitiveHashers.put(float.class, lookup.findStatic(Float.class, "hashCode",
                                                                MethodType.methodType(int.class, float.class)));
            primitiveHashers.put(double.class, lookup.findStatic(Double.class, "hashCode",
                                                                 MethodType.methodType(int.class, double.class)));
            primitiveHashers.put(boolean.class, lookup.findStatic(Boolean.class, "hashCode",
                                                                  MethodType.methodType(int.class, boolean.class)));
        }
        catch (ReflectiveOperationException e) {
            throw new RuntimeException(e);
        }
    }

    private static int hashCombiner(int x, int y) {
        return x*31 + y;
    }

    private static boolean eq(Object a, Object b) { return a == b; }
    private static boolean eq(byte a, byte b) { return a == b; }
    private static boolean eq(short a, short b) { return a == b; }
    private static boolean eq(char a, char b) { return a == b; }
    private static boolean eq(int a, int b) { return a == b; }
    private static boolean eq(long a, long b) { return a == b; }
    private static boolean eq(float a, float b) { return Float.compare(a, b) == 0; }
    private static boolean eq(double a, double b) { return Double.compare(a, b) == 0; }
    private static boolean eq(boolean a, boolean b) { return a == b; }

    /** Get the method handle for combining two values of a given type */
    private static MethodHandle equalator(MethodHandles.Lookup lookup, Class<?> clazz) throws Throwable {
        if (clazz.isPrimitive())
            return primitiveEquals.get(clazz);
        MethodType mt = MethodType.methodType(boolean.class, clazz, clazz);
        return MethodHandles.guardWithTest(IS_ARG0_NULL.asType(mt),
                                           IS_ARG1_NULL.asType(mt),
                                           lookup.findVirtual(clazz, "equals", MT_OBJECT_BOOLEAN).asType(mt));
    }

    /** Get the hasher for a value of a given type */
    private static MethodHandle hasher(MethodHandles.Lookup lookup, Class<?> clazz) throws Throwable {
        if (clazz.isPrimitive())
            return primitiveHashers.get(clazz);
        MethodType mt = MethodType.methodType(int.class, clazz);
        return MethodHandles.guardWithTest(IS_NULL.asType(MethodType.methodType(boolean.class, clazz)),
                                           MethodHandles.dropArguments(MethodHandles.zero(int.class), 0, clazz),
                                           lookup.findVirtual(clazz, "hashCode", MT_INT).asType(mt));
    }

    // If this type must be a monomorphic receiver, that is, one that has no
    // subtypes in the JVM.  For example, Object-typed fields may have a more
    // specific one type at runtime and thus need optimizations.
    private static boolean isMonomorphic(Class<?> type) {
        // Includes primitives and final classes, but not arrays.
        // All array classes are reported to be final, but Object[] can have subtypes like String[]
        return Modifier.isFinal(type.getModifiers()) && !type.isArray();
    }

    private static String specializerClassName(Class<?> targetClass, String kind) {
        String name = targetClass.getName();
        if (targetClass.isHidden()) {
            // use the original class name
            name = name.replace('/', '_');
        }
        return name + "$$" + kind + "Specializer";
    }

    /**
     * Generates a method handle for the {@code equals} method for a given data class
     * @param receiverClass   the data class
     * @param getters         the list of getters
     * @return the method handle
     */
    private static MethodHandle makeEquals(MethodHandles.Lookup lookup, Class<?> receiverClass,
                                           List<MethodHandle> getters) throws Throwable {
        MethodType rr = MethodType.methodType(boolean.class, receiverClass, receiverClass);
        MethodType ro = MethodType.methodType(boolean.class, receiverClass, Object.class);
        MethodHandle instanceFalse = MethodHandles.dropArguments(FALSE, 0, receiverClass, Object.class); // (RO)Z
        MethodHandle instanceTrue = MethodHandles.dropArguments(TRUE, 0, receiverClass, Object.class); // (RO)Z
        MethodHandle isSameObject = OBJECT_EQ.asType(ro); // (RO)Z
        MethodHandle isInstance = MethodHandles.dropArguments(CLASS_IS_INSTANCE.bindTo(receiverClass), 0, receiverClass); // (RO)Z
        MethodHandle accumulator = MethodHandles.dropArguments(TRUE, 0, receiverClass, receiverClass); // (RR)Z

        int size = getters.size();
        MethodHandle[] equalators = new MethodHandle[size];
        boolean hasPolymorphism = false;
        for (int i = 0; i < size; i++) {
            var getter = getters.get(i);
            var type = getter.type().returnType();
            if (isMonomorphic(type)) {
                equalators[i] = equalator(lookup, type);
            } else {
                hasPolymorphism = true;
            }
        }

        // Currently, hotspot does not support polymorphic inlining.
        // As a result, if we have a MethodHandle to Object.equals,
        // it does not enjoy separate profiles like individual invokevirtuals,
        // and we must spin bytecode to accomplish separate profiling.
        if (hasPolymorphism) {
            String[] names = new String[size];

            var classFileContext = ClassFile.of(ClassFile.ClassHierarchyResolverOption.of(ClassHierarchyResolver.ofClassLoading(lookup)));
            var bytes = classFileContext.build(ClassDesc.of(specializerClassName(lookup.lookupClass(), "Equalator")), clb -> {
                for (int i = 0; i < size; i++) {
                    if (equalators[i] == null) {
                        var name = "equalator".concat(Integer.toString(i));
                        names[i] = name;
                        var type = getters.get(i).type().returnType();
                        boolean isInterface = type.isInterface();
                        var typeDesc = type.describeConstable().orElseThrow();
                        clb.withMethodBody(name, MethodTypeDesc.of(CD_boolean, typeDesc, typeDesc), ACC_STATIC, cob -> {
                            var nonNullPath = cob.newLabel();
                            var fail = cob.newLabel();
                            cob.aload(0)
                               .ifnonnull(nonNullPath)
                               .aload(1)
                               .ifnonnull(fail)
                               .iconst_1() // arg0 null, arg1 null
                               .ireturn()
                               .labelBinding(fail)
                               .iconst_0() // arg0 null, arg1 non-null
                               .ireturn()
                               .labelBinding(nonNullPath)
                               .aload(0) // arg0.equals(arg1) - bytecode subject to customized profiling
                               .aload(1)
                               .invoke(isInterface ? Opcode.INVOKEINTERFACE : Opcode.INVOKEVIRTUAL, typeDesc, "equals", MTD_OBJECT_BOOLEAN, isInterface)
                               .ireturn();
                        });
                    }
                }
            });

            var specializerLookup = lookup.defineHiddenClass(bytes, true, MethodHandles.Lookup.ClassOption.STRONG);

            for (int i = 0; i < size; i++) {
                if (equalators[i] == null) {
                    var type = getters.get(i).type().returnType();
                    equalators[i] = specializerLookup.findStatic(specializerLookup.lookupClass(), names[i], MethodType.methodType(boolean.class, type, type));
                }
            }
        }

        for (int i = 0; i < size; i++) {
            var getter = getters.get(i);
            MethodHandle equalator = equalators[i]; // (TT)Z
            MethodHandle thisFieldEqual = MethodHandles.filterArguments(equalator, 0, getter, getter); // (RR)Z
            accumulator = MethodHandles.guardWithTest(thisFieldEqual, accumulator, instanceFalse.asType(rr));
        }

        return MethodHandles.guardWithTest(isSameObject,
                                           instanceTrue,
                                           MethodHandles.guardWithTest(isInstance, accumulator.asType(ro), instanceFalse));
    }

    /**
     * Generates a method handle for the {@code hashCode} method for a given data class
     * @param receiverClass   the data class
     * @param getters         the list of getters
     * @return the method handle
     */
    private static MethodHandle makeHashCode(MethodHandles.Lookup lookup, Class<?> receiverClass,
                                             List<MethodHandle> getters) throws Throwable {
        MethodHandle accumulator = MethodHandles.dropArguments(ZERO, 0, receiverClass); // (R)I

        int size = getters.size();
        MethodHandle[] hashers = new MethodHandle[size];
        boolean hasPolymorphism = false;
        for (int i = 0; i < size; i++) {
            var getter = getters.get(i);
            var type = getter.type().returnType();
            if (isMonomorphic(type)) {
                hashers[i] = hasher(lookup, type);
            } else {
                hasPolymorphism = true;
            }
        }

        // Currently, hotspot does not support polymorphic inlining.
        // As a result, if we have a MethodHandle to Object.hashCode,
        // it does not enjoy separate profiles like individual invokevirtuals,
        // and we must spin bytecode to accomplish separate profiling.
        if (hasPolymorphism) {
            String[] names = new String[size];

            var classFileContext = ClassFile.of(ClassFile.ClassHierarchyResolverOption.of(ClassHierarchyResolver.ofClassLoading(lookup)));
            var bytes = classFileContext.build(ClassDesc.of(specializerClassName(lookup.lookupClass(), "Hasher")), clb -> {
                for (int i = 0; i < size; i++) {
                    if (hashers[i] == null) {
                        var name = "hasher".concat(Integer.toString(i));
                        names[i] = name;
                        var type = getters.get(i).type().returnType();
                        boolean isInterface = type.isInterface();
                        var typeDesc = type.describeConstable().orElseThrow();
                        clb.withMethodBody(name, MethodTypeDesc.of(CD_int, typeDesc), ACC_STATIC, cob -> {
                            var nonNullPath = cob.newLabel();
                            cob.aload(0)
                               .ifnonnull(nonNullPath)
                               .iconst_0() // null hash is 0
                               .ireturn()
                               .labelBinding(nonNullPath)
                               .aload(0) // arg0.hashCode() - bytecode subject to customized profiling
                               .invoke(isInterface ? Opcode.INVOKEINTERFACE : Opcode.INVOKEVIRTUAL, typeDesc, "hashCode", MTD_INT, isInterface)
                               .ireturn();
                        });
                    }
                }
            });

            var specializerLookup = lookup.defineHiddenClass(bytes, true, MethodHandles.Lookup.ClassOption.STRONG);

            for (int i = 0; i < size; i++) {
                if (hashers[i] == null) {
                    var type = getters.get(i).type().returnType();
                    hashers[i] = specializerLookup.findStatic(specializerLookup.lookupClass(), names[i], MethodType.methodType(int.class, type));
                }
            }
        }

        // @@@ Use loop combinator instead?
        for (int i = 0; i < size; i++) {
            var getter = getters.get(i);
            MethodHandle hasher = hashers[i]; // (T)I
            MethodHandle hashThisField = MethodHandles.filterArguments(hasher, 0, getter);    // (R)I
            MethodHandle combineHashes = MethodHandles.filterArguments(HASH_COMBINER, 0, accumulator, hashThisField); // (RR)I
            accumulator = MethodHandles.permuteArguments(combineHashes, accumulator.type(), 0, 0); // adapt (R)I to (RR)I
        }

        return accumulator;
    }

    /**
     * Generates a method handle for the {@code toString} method for a given data class
     * @param receiverClass   the data class
     * @param getters         the list of getters
     * @param names           the names
     * @return the method handle
     */
    private static MethodHandle makeToString(MethodHandles.Lookup lookup,
                                            Class<?> receiverClass,
                                            MethodHandle[] getters,
                                            List<String> names) {
        assert getters.length == names.size();
        if (getters.length == 0) {
            // special case
            MethodHandle emptyRecordCase = MethodHandles.constant(String.class, receiverClass.getSimpleName() + "[]");
            emptyRecordCase = MethodHandles.dropArguments(emptyRecordCase, 0, receiverClass); // (R)S
            return emptyRecordCase;
        }

        boolean firstTime = true;
        MethodHandle[] mhs;
        List<List<MethodHandle>> splits;
        MethodHandle[] toSplit = getters;
        int namesIndex = 0;
        do {
            /* StringConcatFactory::makeConcatWithConstants can only deal with 200 slots, longs and double occupy two
             * the rest 1 slot, we need to chop the current `getters` into chunks, it could be that for records with
             * a lot of components that we need to do a couple of iterations. The main difference between the first
             * iteration and the rest would be on the recipe
             */
            splits = split(toSplit);
            mhs = new MethodHandle[splits.size()];
            for (int splitIndex = 0; splitIndex < splits.size(); splitIndex++) {
                String recipe = "";
                if (firstTime && splitIndex == 0) {
                    recipe = receiverClass.getSimpleName() + "[";
                }
                for (int i = 0; i < splits.get(splitIndex).size(); i++) {
                    recipe += firstTime ? names.get(namesIndex) + "=" + "\1" : "\1";
                    if (firstTime && namesIndex != names.size() - 1) {
                        recipe += ", ";
                    }
                    namesIndex++;
                }
                if (firstTime && splitIndex == splits.size() - 1) {
                    recipe += "]";
                }
                Class<?>[] concatTypeArgs = new Class<?>[splits.get(splitIndex).size()];
                // special case: no need to create another getters if there is only one split
                MethodHandle[] currentSplitGetters = new MethodHandle[splits.get(splitIndex).size()];
                for (int j = 0; j < splits.get(splitIndex).size(); j++) {
                    concatTypeArgs[j] = splits.get(splitIndex).get(j).type().returnType();
                    currentSplitGetters[j] = splits.get(splitIndex).get(j);
                }
                MethodType concatMT = MethodType.methodType(String.class, concatTypeArgs);
                try {
                    mhs[splitIndex] = StringConcatFactory.makeConcatWithConstants(
                            lookup, "",
                            concatMT,
                            recipe,
                            new Object[0]
                    ).getTarget();
                    mhs[splitIndex] = MethodHandles.filterArguments(mhs[splitIndex], 0, currentSplitGetters);
                    // this will spread the receiver class across all the getters
                    mhs[splitIndex] = MethodHandles.permuteArguments(
                            mhs[splitIndex],
                            MethodType.methodType(String.class, receiverClass),
                            new int[splits.get(splitIndex).size()]
                    );
                } catch (Throwable t) {
                    throw new RuntimeException(t);
                }
            }
            toSplit = mhs;
            firstTime = false;
        } while (splits.size() > 1);
        return mhs[0];
    }

    /**
     * Chops the getters into smaller chunks according to the maximum number of slots
     * StringConcatFactory::makeConcatWithConstants can chew
     * @param getters the current getters
     * @return chunks that won't surpass the maximum number of slots StringConcatFactory::makeConcatWithConstants can chew
     */
    private static List<List<MethodHandle>> split(MethodHandle[] getters) {
        List<List<MethodHandle>> splits = new ArrayList<>();

        int slots = 0;

        // Need to peel, so that neither call has more than acceptable number
        // of slots for the arguments.
        List<MethodHandle> cArgs = new ArrayList<>();
        for (MethodHandle methodHandle : getters) {
            Class<?> returnType = methodHandle.type().returnType();
            int needSlots = (returnType == long.class || returnType == double.class) ? 2 : 1;
            if (slots + needSlots > MAX_STRING_CONCAT_SLOTS) {
                splits.add(cArgs);
                cArgs = new ArrayList<>();
                slots = 0;
            }
            cArgs.add(methodHandle);
            slots += needSlots;
        }

        // Flush the tail slice
        if (!cArgs.isEmpty()) {
            splits.add(cArgs);
        }

        return splits;
    }

    /**
     * Bootstrap method to generate the {@link Object#equals(Object)},
     * {@link Object#hashCode()}, and {@link Object#toString()} methods, based
     * on a description of the component names and accessor methods, for either
     * {@code invokedynamic} call sites or dynamic constant pool entries.
     *
     * For more detail on the semantics of the generated methods see the specification
     * of {@link java.lang.Record#equals(Object)}, {@link java.lang.Record#hashCode()} and
     * {@link java.lang.Record#toString()}.
     *
     *
     * @param lookup       Every bootstrap method is expected to have a {@code lookup}
     *                     which usually represents a lookup context with the
     *                     accessibility privileges of the caller. This is because
     *                     {@code invokedynamic} call sites always provide a {@code lookup}
     *                     to the corresponding bootstrap method, but this method just
     *                     ignores the {@code lookup} parameter
     * @param methodName   the name of the method to generate, which must be one of
     *                     {@code "equals"}, {@code "hashCode"}, or {@code "toString"}
     * @param type         a {@link MethodType} corresponding the descriptor type
     *                     for the method, which must correspond to the descriptor
     *                     for the corresponding {@link Object} method, if linking
     *                     an {@code invokedynamic} call site, or the
     *                     constant {@code MethodHandle.class}, if linking a
     *                     dynamic constant
     * @param recordClass  the record class hosting the record components
     * @param names        the list of component names, joined into a string
     *                     separated by ";", or the empty string if there are no
     *                     components. This parameter is ignored if the {@code methodName}
     *                     parameter is {@code "equals"} or {@code "hashCode"}
     * @param getters      method handles for the accessor methods for the components
     * @return             a call site if invoked by indy, or a method handle
     *                     if invoked by a condy
     * @throws IllegalArgumentException if the bootstrap arguments are invalid
     *                                  or inconsistent
     * @throws NullPointerException if any argument is {@code null} or if any element
     *                              in the {@code getters} array is {@code null}
     * @throws Throwable if any exception is thrown during call site construction
     */
    public static Object bootstrap(MethodHandles.Lookup lookup, String methodName, TypeDescriptor type,
                                   Class<?> recordClass,
                                   String names,
                                   MethodHandle... getters) throws Throwable {
        requireNonNull(lookup);
        requireNonNull(methodName);
        requireNonNull(type);
        requireNonNull(recordClass);
        requireNonNull(names);
        requireNonNull(getters);
        Arrays.stream(getters).forEach(Objects::requireNonNull);
        MethodType methodType;
        if (type instanceof MethodType mt)
            methodType = mt;
        else {
            methodType = null;
            if (!MethodHandle.class.equals(type))
                throw new IllegalArgumentException(type.toString());
        }
        List<MethodHandle> getterList = List.of(getters);
        MethodHandle handle = switch (methodName) {
            case "equals"   -> {
                if (methodType != null && !methodType.equals(MethodType.methodType(boolean.class, recordClass, Object.class)))
                    throw new IllegalArgumentException("Bad method type: " + methodType);
                yield makeEquals(lookup, recordClass, getterList);
            }
            case "hashCode" -> {
                if (methodType != null && !methodType.equals(MethodType.methodType(int.class, recordClass)))
                    throw new IllegalArgumentException("Bad method type: " + methodType);
                yield makeHashCode(lookup, recordClass, getterList);
            }
            case "toString" -> {
                if (methodType != null && !methodType.equals(MethodType.methodType(String.class, recordClass)))
                    throw new IllegalArgumentException("Bad method type: " + methodType);
                List<String> nameList = "".equals(names) ? List.of() : List.of(names.split(";"));
                if (nameList.size() != getterList.size())
                    throw new IllegalArgumentException("Name list and accessor list do not match");
                yield makeToString(lookup, recordClass, getters, nameList);
            }
            default -> throw new IllegalArgumentException(methodName);
        };
        return methodType != null ? new ConstantCallSite(handle) : handle;
    }
}
